// SClawHub Content Script - Injects trust badges on ClawHub

const API_BASE = 'https://sclawhub.com';
const cache = new Map();

function extractSkillSlug(url) {
  const match = url.match(/\/skills\/([^\/\?#]+)/);
  return match ? match[1] : null;
}

function createBadge(score, riskLevel) {
  const badge = document.createElement('span');
  badge.className = 'sclawhub-badge';
  
  let emoji, color, bgColor;
  
  if (score >= 90) {
    emoji = '🛡️';
    color = '#10b981';
    bgColor = 'rgba(16, 185, 129, 0.1)';
  } else if (score >= 70) {
    emoji = '✅';
    color = '#84cc16';
    bgColor = 'rgba(132, 204, 22, 0.1)';
  } else if (score >= 50) {
    emoji = '⚠️';
    color = '#eab308';
    bgColor = 'rgba(234, 179, 8, 0.1)';
  } else if (score >= 30) {
    emoji = '⚠️';
    color = '#f97316';
    bgColor = 'rgba(249, 115, 22, 0.1)';
  } else {
    emoji = '🚨';
    color = '#ef4444';
    bgColor = 'rgba(239, 68, 68, 0.1)';
  }
  
  badge.innerHTML = `
    <span style="
      display: inline-flex;
      align-items: center;
      gap: 4px;
      padding: 4px 10px;
      border-radius: 12px;
      font-size: 12px;
      font-weight: 600;
      color: ${color};
      background: ${bgColor};
      border: 1px solid ${color}40;
      cursor: pointer;
      margin-left: 8px;
    ">
      ${emoji} ${score}/100
    </span>
  `;
  
  return badge;
}

function createNotScannedBadge() {
  const badge = document.createElement('span');
  badge.className = 'sclawhub-badge-not-scanned';
  
  badge.innerHTML = `
    <span style="
      display: inline-flex;
      align-items: center;
      gap: 4px;
      padding: 4px 10px;
      border-radius: 12px;
      font-size: 12px;
      font-weight: 600;
      color: #94a3b8;
      background: rgba(148, 163, 184, 0.1);
      border: 1px solid #94a3b840;
      cursor: pointer;
      margin-left: 8px;
    ">
      ❓ Scan
    </span>
  `;
  
  return badge;
}

async function fetchTrustScore(slug) {
  // Check cache first
  if (cache.has(slug)) {
    return cache.get(slug);
  }
  
  try {
    const response = await fetch(`${API_BASE}/api/skills/${slug}`);
    
    if (response.ok) {
      const data = await response.json();
      cache.set(slug, data);
      return data;
    }
    
    return null;
  } catch (error) {
    console.error('SClawHub: Failed to fetch trust score:', error);
    return null;
  }
}

function injectBadges() {
  // Find all skill links on the page
  const skillLinks = document.querySelectorAll('a[href*="/skills/"]');
  
  skillLinks.forEach(async (link) => {
    // Skip if already processed
    if (link.querySelector('.sclawhub-badge, .sclawhub-badge-not-scanned')) {
      return;
    }
    
    const slug = extractSkillSlug(link.href);
    if (!slug) return;
    
    // Fetch trust score
    const report = await fetchTrustScore(slug);
    
    if (report) {
      const badge = createBadge(report.trustScore, report.riskLevel);
      
      // Add click handler to open full report
      badge.addEventListener('click', (e) => {
        e.preventDefault();
        e.stopPropagation();
        window.open(`${API_BASE}/skills/${slug}`, '_blank');
      });
      
      link.appendChild(badge);
    } else {
      const badge = createNotScannedBadge();
      
      // Add click handler to open scan page
      badge.addEventListener('click', (e) => {
        e.preventDefault();
        e.stopPropagation();
        window.open(`${API_BASE}/scan`, '_blank');
      });
      
      link.appendChild(badge);
    }
  });
}

// Run on page load
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', injectBadges);
} else {
  injectBadges();
}

// Re-run when new content is added (for SPAs)
const observer = new MutationObserver((mutations) => {
  let shouldInject = false;
  
  mutations.forEach((mutation) => {
    if (mutation.addedNodes.length > 0) {
      shouldInject = true;
    }
  });
  
  if (shouldInject) {
    injectBadges();
  }
});

observer.observe(document.body, {
  childList: true,
  subtree: true
});

console.log('🦞🛡️ SClawHub extension loaded');
